<?php

namespace App\Services;

use App\Models\TelegramUser;
use App\Models\UserInvestment;
use Telegram\Bot\Api;
use Telegram\Bot\Keyboard\Keyboard;

class TelegramBotService
{
    protected Api $telegram;

    public function __construct()
    {
        $this->telegram = new Api(env('TELEGRAM_BOT_TOKEN'));

        // SSL verification disable for development
        $client = new \GuzzleHttp\Client(['verify' => false]);
        $this->telegram->setHttpClientHandler(
            new \Telegram\Bot\HttpClients\GuzzleHttpClient($client)
        );
    }

    public function handleUpdate($update)
    {
        if (!$update->getMessage()) {
            return;
        }

        $message = $update->getMessage();
        $chatId = $message->getChat()->getId();
        $text = $message->getText();
        $from = $message->getFrom();

        // User ko database mein save/update karo
        $user = $this->getOrCreateUser($chatId, $from);

        if ($user->is_blocked) {
            return $this->sendMessage($chatId, "⛔ You are blocked from using this bot.");
        }

        // Admin commands
        if ($user->isAdmin()) {
            return $this->handleAdminCommands($chatId, $text, $user, $message);
        }

        // Regular user commands
        return $this->handleUserCommands($chatId, $text, $user, $message);
    }

    protected function getOrCreateUser($chatId, $from)
    {
        return TelegramUser::firstOrCreate(
            ['chat_id' => $chatId],
            [
                'username' => $from->getUsername(),
                'first_name' => $from->getFirstName(),
                'last_name' => $from->getLastName(),
            ]
        );
    }

    protected function handleUserCommands($chatId, $text, $user, $message)
    {
        switch ($user->current_state) {
            case 'awaiting_investment_amount':
                return $this->handleInvestmentAmount($chatId, $text, $user);

            case 'awaiting_transaction_proof':
                return $this->handleTransactionProof($chatId, $user, $message);

            default:
                return $this->handleMainMenu($chatId, $text, $user);
        }
    }

    protected function handleMainMenu($chatId, $text, $user)
    {
        switch ($text) {
            case '/start':
                return $this->showMainMenu($chatId);

            case '💰 New Investment':
                return $this->startInvestmentProcess($chatId, $user);

            case '📊 My Investments':
                return $this->showUserInvestments($chatId, $user);

            case '💼 Investment Plans':
                return $this->showInvestmentPlans($chatId);

            case '📞 Contact Support':
                return $this->showSupport($chatId);

            case '❓ Help':
                return $this->showHelp($chatId);

            default:
                return $this->showMainMenu($chatId);
        }
    }

    protected function showMainMenu($chatId)
    {
        $keyboard = Keyboard::make([
            'keyboard' => [
                ['💰 New Investment', '📊 My Investments'],
                ['💼 Investment Plans', '📞 Contact Support'],
                ['❓ Help']
            ],
            'resize_keyboard' => true,
            'one_time_keyboard' => false
        ]);

        return $this->sendMessage(
            $chatId,
            "🌟 *Welcome to Elite Crypto Investment Company Bot* 🌟\n\n" .
                "💎 Start your crypto investment journey today!\n\n" .
                "Please select an option from the menu below:",
            $keyboard
        );
    }

    protected function startInvestmentProcess($chatId, $user)
    {
        $user->updateState('awaiting_investment_amount');

        $keyboard = Keyboard::make([
            'keyboard' => [
                ['🔙 Back to Menu']
            ],
            'resize_keyboard' => true
        ]);

        return $this->sendMessage(
            $chatId,
            "💰 *New Investment*\n\n" .
                "Please enter the amount you want to invest (in INR):\n\n" .
                "Example: 3000 or 5000 ",
            $keyboard
        );
    }

    protected function handleInvestmentAmount($chatId, $text, $user)
    {
        if ($text === '🔙 Back to Menu') {
            $user->updateState('main_menu');
            return $this->showMainMenu($chatId);
        }

        $amount = floatval($text);

        if ($amount < 3000) {
            return $this->sendMessage(
                $chatId,
                "❌ Minimum investment amount is ₹3000 .\n" .
                    "Please enter a valid amount:"
            );
        }

        // Create investment record
        $investment = UserInvestment::create([
            'telegram_user_id' => $user->id,
            'amount' => $amount,
            'status' => 'pending'
        ]);

        $user->updateState('awaiting_transaction_proof');

        return $this->sendMessage(
            $chatId,
            "✅ Investment Amount: \$$amount USD\n\n" .
                "📤 Please send your transaction proof (screenshot/photo)\n\n" .
                "💳 Payment Details:\n" .
                "UPI ID: `elitefund@upi`\n\n" .
                "Bitcoin: `bc1qxy2kgdygjrsqtzq2n0yrf2493p83kkfjhx0wlh`\n" .
                "USDT (TRC20) : `TMWo731ShDWjS5E3AWLEPjoKaD8Zgy3X35`\n\n" .
                "After payment, send the screenshot here."
        );
    }

    protected function handleTransactionProof($chatId, $user, $message)
    {
        if ($message->getText() === '🔙 Back to Menu') {
            $user->updateState('main_menu');
            return $this->showMainMenu($chatId);
        }

        if ($message->getPhoto()) {
            $photo = $message->getPhoto()[count($message->getPhoto()) - 1];
            $fileId = $photo->getFileId();

            // Update latest investment
            $investment = $user->investments()->latest()->first();
            if ($investment) {
                $investment->update(['transaction_proof' => $fileId]);
            }

            $user->updateState('main_menu');

            $this->sendMessage(
                $chatId,
                "✅ *Transaction proof received!*\n\n" .
                    "🔍 Your investment is under review.\n" .
                    "📱 We will notify you once approved.\n\n" .
                    "Thank you for choosing Elite Crypto! 🚀"
            );

            // Notify admin
            $this->notifyAdminNewInvestment($investment);

            return $this->showMainMenu($chatId);
        }

        return $this->sendMessage(
            $chatId,
            "❌ Please send a photo/screenshot of your transaction proof."
        );
    }

    protected function showUserInvestments($chatId, $user)
    {
        $investments = $user->investments()->latest()->get();

        if ($investments->isEmpty()) {
            return $this->sendMessage(
                $chatId,
                "📊 *My Investments*\n\n" .
                    "You don't have any investments yet.\n\n" .
                    "Start investing now! 💰"
            );
        }

        $message = "📊 *Your Investments*\n\n";

        foreach ($investments as $investment) {
            $statusEmoji = [
                'pending' => '⏳',
                'approved' => '✅',
                'completed' => '🎉'
            ];

            $message .= "━━━━━━━━━━━━━━━\n";
            $message .= "💵 Amount: \₹{$investment->amount}\n";
            $message .= "📅 Date: " . $investment->created_at->format('d M Y') . "\n";
            $message .= "📈 Status: {$statusEmoji[$investment->status]} " . ucfirst($investment->status) . "\n";
        }

        return $this->sendMessage($chatId, $message);
    }

    protected function showInvestmentPlans($chatId)
    {
        $message = "💼 *Investment Plans*\n\n";
        $message .= "━━━━━━━━━━━━━━━\n\n";

        $message .= "🥉 *STARTER PLAN*\n";
        $message .= "💰 Min: \₹3000 - \₹10,000\n";
        $message .= "📈 ROI: 20k-30k daily\n";
        $message .= "⏱ Duration: 30 Minutes\n\n";

        $message .= "🥈 *PREMIUM PLAN*\n";
        $message .= "💰 Min: \₹11,000 - \₹20,000\n";
        $message .= "📈 ROI: 40k-55k daily\n";
        $message .= "⏱ Duration: 30 Minutes\n\n";

        $message .= "🥇 *VIP PLAN*\n";
        $message .= "💰 Min: \₹21,000+\n";
        $message .= "📈 ROI: 60k-1Lakh+ daily\n";
        $message .= "⏱ Duration: 30 Minutes\n\n";

        $message .= "✨ All plans include:\n";
        $message .= "• 24/7 Support\n";
        $message .= "• Instant Withdrawals\n";
        $message .= "• Secure Transactions\n";

        return $this->sendMessage($chatId, $message);
    }

    protected function showSupport($chatId)
    {
        return $this->sendMessage(
            $chatId,
            "📞 *Contact Support*\n\n" .
                "💬 Telegram: @elitecrypto_support\n" .
                "📧 Email: elitecryptofund@proton.me\n" .
                "🌐 Website: www.elitecrypto.com\n\n" .
                "We're here to help 24/7! 🚀"
        );
    }

    protected function showHelp($chatId)
    {
        return $this->sendMessage(
            $chatId,
            "❓ *Help & FAQ*\n\n" .
                "🔹 How to invest?\n" .
                "Tap '💰 New Investment' and follow the steps.\n\n" .
                "🔹 How to track investments?\n" .
                "Tap '📊 My Investments' to see your portfolio.\n\n" .
                "🔹 Payment methods?\n" .
                "We accept UPI, BTC, USDT, and other cryptocurrencies.\n\n" .
                "🔹 When will I get profits?\n" .
                "Daily profits start after investment approval.\n\n" .
                "Need more help? Contact support! 📞"
        );
    }

    // Admin Functions
    protected function handleAdminCommands($chatId, $text, $user, $message)
    {
        if ($text === '/admin') {
            return $this->showAdminPanel($chatId);
        }

        if ($text === '📊 Statistics') {
            return $this->showStatistics($chatId);
        }

        if ($text === '👥 All Users') {
            return $this->showAllUsers($chatId);
        }

        if ($text === '⏳ Pending Investments') {
            return $this->showPendingInvestments($chatId);
        }

        if ($text === '📢 Broadcast Message') {
            $user->updateState('awaiting_broadcast_message');
            return $this->sendMessage(
                $chatId,
                "📢 *Broadcast Message*\n\n" .
                    "Send the message you want to broadcast to all users:\n\n" .
                    "You can send:\n" .
                    "• Text message\n" .
                    "• Photo with caption\n" .
                    "• Video with caption"
            );
        }

        if ($user->current_state === 'awaiting_broadcast_message') {
            return $this->handleBroadcast($chatId, $user, $message);
        }

        // Handle approve/reject commands
        if (strpos($text, '/approve_') === 0 || strpos($text, '/reject_') === 0) {
            return $this->handleInvestmentApproval($chatId, $text);
        }

        return $this->handleMainMenu($chatId, $text, $user);
    }

    protected function showAdminPanel($chatId)
    {
        $keyboard = Keyboard::make([
            'keyboard' => [
                ['📊 Statistics', '👥 All Users'],
                ['⏳ Pending Investments', '📢 Broadcast Message'],
                ['🔙 Back to Menu']
            ],
            'resize_keyboard' => true
        ]);

        return $this->sendMessage(
            $chatId,
            "🔐 *Admin Panel*\n\n" .
                "Select an option:",
            $keyboard
        );
    }

    protected function showStatistics($chatId)
    {
        $totalUsers = TelegramUser::count();
        $totalInvestments = UserInvestment::sum('amount');
        $pendingInvestments = UserInvestment::where('status', 'pending')->count();
        $approvedInvestments = UserInvestment::where('status', 'approved')->count();

        return $this->sendMessage(
            $chatId,
            "📊 *Bot Statistics*\n\n" .
                "👥 Total Users: {$totalUsers}\n" .
                "💰 Total Investments: \${$totalInvestments}\n" .
                "⏳ Pending: {$pendingInvestments}\n" .
                "✅ Approved: {$approvedInvestments}"
        );
    }

    protected function showAllUsers($chatId)
    {
        $users = TelegramUser::latest()->take(20)->get();

        $message = "👥 *Recent Users (Last 20)*\n\n";

        foreach ($users as $user) {
            $message .= "━━━━━━━━━━━━━━━\n";
            $message .= "👤 {$user->first_name}\n";
            $message .= "🆔 Chat ID: `{$user->chat_id}`\n";
            $message .= "📅 Joined: " . $user->created_at->format('d M Y') . "\n";
        }

        return $this->sendMessage($chatId, $message);
    }

    protected function showPendingInvestments($chatId)
    {
        $investments = UserInvestment::with('user')
            ->where('status', 'pending')
            ->latest()
            ->get();

        if ($investments->isEmpty()) {
            return $this->sendMessage($chatId, "✅ No pending investments!");
        }

        foreach ($investments as $investment) {
            $message = "⏳ *Pending Investment #{$investment->id}*\n\n";
            $message .= "👤 User: {$investment->user->first_name}\n";
            $message .= "🆔 Chat ID: `{$investment->user->chat_id}`\n";
            $message .= "💰 Amount: \${$investment->amount}\n";
            $message .= "📅 Date: " . $investment->created_at->format('d M Y H:i') . "\n\n";
            $message .= "To approve: /approve_{$investment->id}\n";
            $message .= "To reject: /reject_{$investment->id}";

            $this->sendMessage($chatId, $message);

            // Send transaction proof if available
            if ($investment->transaction_proof) {
                $this->telegram->sendPhoto([
                    'chat_id' => $chatId,
                    'photo' => $investment->transaction_proof,
                    'caption' => "Transaction Proof for Investment #{$investment->id}"
                ]);
            }
        }

        return true;
    }

    protected function handleInvestmentApproval($chatId, $text)
    {
        $parts = explode('_', $text);
        $action = str_replace('/', '', $parts[0]);
        $investmentId = $parts[1] ?? null;

        if (!$investmentId) {
            return $this->sendMessage($chatId, "❌ Invalid command format!");
        }

        $investment = UserInvestment::find($investmentId);

        if (!$investment) {
            return $this->sendMessage($chatId, "❌ Investment not found!");
        }

        if ($action === 'approve') {
            $investment->update(['status' => 'approved']);

            // Notify user
            $this->sendMessage(
                $investment->user->chat_id,
                "🎉 *Investment Approved!*\n\n" .
                    "💰 Amount: \₹{$investment->amount}\n" .
                    "✅ Your investment has been approved!\n" .
                    "📈 You will start receiving daily returns.\n\n" .
                    "Thank you for trusting Elite Crypto! 🚀"
            );

            return $this->sendMessage(
                $chatId,
                "✅ Investment #{$investmentId} approved!\n" .
                    "User has been notified."
            );
        } else {
            $investment->update(['status' => 'rejected']);

            // Notify user
            $this->sendMessage(
                $investment->user->chat_id,
                "❌ *Investment Rejected*\n\n" .
                    "💰 Amount: \₹{$investment->amount}\n" .
                    "Unfortunately, your investment could not be verified.\n\n" .
                    "📞 Please contact support for assistance."
            );

            return $this->sendMessage(
                $chatId,
                "❌ Investment #{$investmentId} rejected!\n" .
                    "User has been notified."
            );
        }
    }

    protected function handleBroadcast($chatId, $admin, $message)
    {
        $admin->updateState('main_menu');

        $text = $message->getText();
        $photo = $message->getPhoto();
        $video = $message->getVideo();

        $users = TelegramUser::where('is_blocked', false)
            ->where('chat_id', '!=', $chatId)
            ->get();

        $sent = 0;
        $failed = 0;

        foreach ($users as $user) {
            try {
                if ($photo) {
                    $photoFile = $photo[count($photo) - 1];
                    $this->telegram->sendPhoto([
                        'chat_id' => $user->chat_id,
                        'photo' => $photoFile->getFileId(),
                        'caption' => $message->getCaption() ?? ''
                    ]);
                } elseif ($video) {
                    $this->telegram->sendVideo([
                        'chat_id' => $user->chat_id,
                        'video' => $video->getFileId(),
                        'caption' => $message->getCaption() ?? ''
                    ]);
                } else {
                    $this->sendMessage($user->chat_id, $text);
                }
                $sent++;
            } catch (\Exception $e) {
                $failed++;
            }
        }

        $this->sendMessage(
            $chatId,
            "✅ *Broadcast Complete!*\n\n" .
                "📤 Sent: {$sent}\n" .
                "❌ Failed: {$failed}"
        );

        return $this->showAdminPanel($chatId);
    }

    protected function notifyAdminNewInvestment($investment)
    {
        $admins = TelegramUser::where('is_admin', true)->get();

        foreach ($admins as $admin) {
            $message = "🔔 *New Investment Alert!*\n\n";
            $message .= "👤 User: {$investment->user->first_name}\n";
            $message .= "💰 Amount: \₹{$investment->amount}\n";
            $message .= "📅 Date: " . $investment->created_at->format('d M Y H:i') . "\n\n";
            $message .= "Check /admin panel for details.";

            $this->sendMessage($admin->chat_id, $message);
        }
    }

    public function sendMessage($chatId, $text, $keyboard = null)
    {
        $params = [
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => 'Markdown'
        ];

        if ($keyboard) {
            $params['reply_markup'] = $keyboard;
        }

        return $this->telegram->sendMessage($params);
    }

    public function sendMessageToUser($chatId, $text, $photo = null, $video = null)
    {
        try {
            if ($photo) {
                $photoPath = storage_path('app/' . $photo);
                return $this->telegram->sendPhoto([
                    'chat_id' => $chatId,
                    'photo' => fopen($photoPath, 'r'),
                    'caption' => $text,
                    'parse_mode' => 'Markdown'
                ]);
            } elseif ($video) {
                $videoPath = storage_path('app/' . $video);
                return $this->telegram->sendVideo([
                    'chat_id' => $chatId,
                    'video' => fopen($videoPath, 'r'),
                    'caption' => $text,
                    'parse_mode' => 'Markdown'
                ]);
            } else {
                return $this->sendMessage($chatId, $text);
            }
        } catch (\Exception $e) {
            return false;
        }
    }
}